#include "DMD_Semesin.h"

/*
  Uncomment the following line if you don't want DMD library to touch
  any timer functionality (ie if you want to use TimerOne library or
  similar, or other libraries that use timers.

  With timers enabled, on AVR-based Arduinos this code will register a
  Timer1 overflow handler (without disrupting any built-in library
  functions.) On ARM-based Arduino Due it will use use Timer7.

*/

//#define NO_TIMERS

#define ESP8266_TIMER0_TICKS microsecondsToClockCycles(1000) // 1000 microseconds between calls to scan_running_dmds seems to work well.

#ifdef NO_TIMERS

// Timer-free stub code which gets compiled in only if NO_TIMERS is set
void HUB12::begin() {
  beginNoTimer();
}

void HUB12::end() {
}

#else // Use timers

// Forward declarations for tracking currently running DMDs
// static void register_running_dmd(HUB12 *dmd);
// static bool unregister_running_dmd(HUB12 *dmd);
// static void inline scan_running_dmds();

static DMDFrame *singleDMDFrame;
static volatile HUB12 **running_dmds = 0;
static volatile int running_dmd_len = 0;

#ifdef __AVR__

/* This AVR timer ISR uses the standard /64 timing used by Timer1 in the Arduino core,
   so none of those registers (or normal PWM timing) is changed. We do skip 50% of ISRs
   as 50% timer overflows is approximately every 4ms, which is fine for flicker-free
   updating.
*/
ISR(TIMER1_OVF_vect)
{
  singleDMDFrame->scanDisplay();
}

void HUB12::begin()
{
  beginNoTimer(); // Do any generic setup

  singleDMDFrame = this;
  TIMSK1 = _BV(TOIE1); // set overflow interrupt

}

void HUB12::end()
{
  TIMSK1 &= ~_BV(TOIE1); // disable timer interrupt, no more DMDs are running
  clearScreen();
  scanDisplay();
}


#elif defined (ESP8266)

void ICACHE_RAM_ATTR handler_timerDMD(){
    singleDMDFrame->scanDisplay();
	timer0_write(ESP.getCycleCount() + ESP8266_TIMER0_TICKS);
}

void HUB12::begin()
{
  beginNoTimer();
  singleDMDFrame = this;
  timer0_detachInterrupt();
  
  timer0_isr_init();
  timer0_attachInterrupt(handler_timerDMD);
  timer0_write(ESP.getCycleCount() + ESP8266_TIMER0_TICKS);
}

void HUB12::end()
{
  timer0_detachInterrupt();
  clearScreen();
  scanDisplay();
}

// #elif defined (MCU_STM32F103C8) || defined (MCU_STM32F103CB) //bluepill, maple
#elif STM32_MCU_SERIES == STM32_SERIES_F1

//timer 2 interrupt

volatile bool toggle;
void handler_timerDMD(void) {

  singleDMDFrame->scanDisplay();

}

void HUB12::begin()
{
  beginNoTimer(); // Do any generic setup
  singleDMDFrame = this;

  //timer 2 setup
  TIMER1_BASE->PSC = 7200; //72MHz/7200=10,000 Hz
  TIMER1_BASE->ARR = 5; //10000/10000=1Hz
  TIMER1_BASE->CNT = 0;  //clear counter

  timer_attach_interrupt(TIMER1, 0, handler_timerDMD); //interrupt on timer update

  TIMER1_BASE->CR1 |= 0x0001; //enable timer.
}

void HUB12::end()
{
  TIMER1_BASE->CR1 &= ~0x0001;
  timer_detach_interrupt(TIMER1, 0);
}

#elif defined (__arm__) // __ARM__, Due assumed for now

/* ARM timer callback (ISR context), checks timer status then scans all running DMDs */
void TC7_Handler() {
  TC_GetStatus(TC2, 1);
  singleDMDFrame->scanDisplay();
}

void HUB12::begin()
{
  beginNoTimer(); // Do any generic setup

  NVIC_DisableIRQ(TC7_IRQn);

  singleDMDFrame = this;

  pmc_set_writeprotect(false);
  pmc_enable_periph_clk(TC7_IRQn);
  // Timer 7 is TC2, channel 1
  TC_Configure(TC2, 1, TC_CMR_WAVE | TC_CMR_WAVSEL_UP_RC | TC_CMR_TCCLKS_TIMER_CLOCK4); // counter up, /128 divisor
  TC_SetRC(TC2, 1, 2500); // approx 4ms at /128 divisor
  TC2->TC_CHANNEL[1].TC_IER = TC_IER_CPCS;

  NVIC_ClearPendingIRQ(TC7_IRQn);
  NVIC_EnableIRQ(TC7_IRQn);
  TC_Start(TC2, 1);
}

void HUB12::end()
{
  NVIC_DisableIRQ(TC7_IRQn);
  TC_Stop(TC2, 1);
}

#endif // ifdef __AVR__

#endif // ifdef NO_TIMERS
